/*
 * Error handling routines.  A stack is maintained for each thread of
 * errors which occur.
 */

#include <pthread.h>
#include <stdlib.h>

#include "libfma.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_notify.h"

/*
 * Private struct so not everyone has to include pthread.h
 */

/*
 * structure for maintaining a stack of error information
 */
struct fms_error_stack {
  int error_index;
  char error_str[FMS_MAX_ERROR_DEPTH][LF_STRING_LEN];
  char *error_file[FMS_MAX_ERROR_DEPTH];
  int error_line[FMS_MAX_ERROR_DEPTH];
  int error_errno[FMS_MAX_ERROR_DEPTH];
};

struct fms_error_vars {
  pthread_key_t key;		/* key for referencing error stack */
};

/*
 * Allocate error vars
 */
void
fms_init_error_vars()
{
  struct fms_error_vars *evp;
  int rc;

  /* Allocate space for vars */
  evp = (struct fms_error_vars *) calloc(sizeof(*evp), 1);
  if (evp == NULL) {
    fprintf(stderr, "Error allocating memory for error subsystem\n");
    fms_exit(1);
  }

  /* save this away */
  F.error_vars = evp;

  /* allocate key for threads to find their error stack */
  rc = pthread_key_create(&(evp->key), free);
  if (rc != 0) {
    perror("creating error key");
    fms_exit(1);
  }
}


/*
 * Allocate the error stack for this thread.  Must be called at the start of
 * each thread.
 */
void
fms_init_error()
{
  struct fms_error_stack *errp;
  int rc;

  errp = (struct fms_error_stack *) calloc(sizeof(*errp), 1);
  if (errp == NULL) {
    perror("Allocating error stack");
    fms_exit(1);
  }

  /* stash this away in thread-specific data */
  rc = pthread_setspecific(F.error_vars->key, errp);
  if (rc != 0) {
    perror("pthread_setspecific");
    fms_exit(1);
  }
}


/*
 * Print the error stack
 */
void
fms_perror()
{
  int i;
  char buf[2048];
  char *bp;
  struct fms_error_stack *errp;

  /*
   * Get the thread-specific error struct for this thread
   */
  errp = (struct fms_error_stack *) pthread_getspecific(F.error_vars->key);
  if (errp == NULL) {
    fprintf(stderr, "Fatal error in error handling! Help!\n");
    fms_exit(1);
  }

  bp = buf;
  for (i=0; i<errp->error_index; ++i) {
    if (i > 0) bp += sprintf(bp, "\n");

    bp += sprintf(bp, "%s:%d %s", errp->error_file[i], errp->error_line[i],
            errp->error_str[i]);
    if (errp->error_errno[i] != 0) {
      bp += sprintf(bp, ": %s", strerror(errp->error_errno[i]));
    }
  }

  fms_notify(FMS_EVENT_ERROR, buf);
  fms_reset_errors();
}

/*
 * save away an error message
 */
void
fms_save_error(
  char *file,
  int lineno,
  int err_no,
  char *errstr)
{
  struct fms_error_stack *errp;

  /*
   * Get the thread-specific error struct for this thread
   */
  errp = (struct fms_error_stack *) pthread_getspecific(F.error_vars->key);
  if (errp == NULL) {
    fprintf(stderr, "Fatal error in error handling! Help!\n");
    fms_exit(1);
  }

  if (errp->error_index >= (FMS_MAX_ERROR_DEPTH-1)) {
    strncpy(errp->error_str[FMS_MAX_ERROR_DEPTH-1], "Error depth overflow",
	LF_STRING_LEN-1);
    errp->error_index = FMS_MAX_ERROR_DEPTH;
  } else {
    errp->error_file[errp->error_index] = file;
    errp->error_line[errp->error_index] = lineno;
    errp->error_errno[errp->error_index] = err_no;

    /* error_str[] is declared with LF_STRING_LEN */
    strncpy(errp->error_str[errp->error_index++], errstr, LF_STRING_LEN-1);
  }					
}

/*
 * Reset all errors
 */
void
fms_reset_errors()
{
  struct fms_error_stack *errp;

  /*
   * Get the thread-specific error struct for this thread
   */
  errp = (struct fms_error_stack *) pthread_getspecific(F.error_vars->key);
  if (errp == NULL) {
    fprintf(stderr, "Fatal error in error handling! Help!\n");
    fms_exit(1);
  }

  errp->error_index = 0;
}

/*
 * Copy out the top error string
 */
void
fms_error_copy(
  char *dest)
{
  struct fms_error_stack *errp;
  int index;

  /*
   * Get the thread-specific error struct for this thread
   */
  errp = (struct fms_error_stack *) pthread_getspecific(F.error_vars->key);
  if (errp == NULL) {
    fprintf(stderr, "Fatal error in error handling! Help!\n");
    fms_exit(1);
  }

  index = errp->error_index;
  if (index > 0) {
    strcpy(dest, errp->error_str[index-1]);
  } else {
    strcpy(dest, "No error");
  }
}
